﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Text;
using FyndSharp.Utilities.IO;
using FyndSharp.Utilities.Serialization;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Utils;
using gov.va.med.vbecs.Common.Log;

namespace gov.va.med.VBECS.Communication.Protocols
{
    /// <summary>
    /// Protocol for communication with HL7 side.
    /// This protocol doesn't preserve message type information.
    /// It deserealizes only RawMessage.
    /// </summary>
// ReSharper disable InconsistentNaming
    public class MLLPProtocol : IProtocol
// ReSharper restore InconsistentNaming
    {
        // This MemoryStream object is used to collect receiving bytes to build messages.
        private MemoryStream _receiveMemoryStream = new MemoryStream();
        // This is special logger for communication tracing
        private readonly ILogger _traceLogger =
            LogManager.Instance().LoggerLocator.GetLogger("CommunicationTrace");
        // Regular logger
        private readonly ILogger _logger =
            LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);
        // Messages serializer
        private readonly IMessageSerializer _sr;
        // Messages factory
        private readonly IMessageFactory _messagesFactory = new RawDataMessageFactory();

        /// <summary>
        /// Constructor
        /// </summary>
        public MLLPProtocol()
        {
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="sr">Message serializer</param>
        /// <param name="messagesFactory">Messages factory deserializes messages from bytes</param>
        public MLLPProtocol(IMessageSerializer sr, IMessageFactory messagesFactory)
        {
            _sr = sr;
            _messagesFactory = messagesFactory;
        }

        /// <summary>
        /// Converts message content to bytes array
        /// </summary>
        /// <param name="theMsg"></param>
        /// <returns>Bytes</returns>
        /// <exception cref="ArgumentNullException">Message object</exception>
        public virtual byte[] GetBytes(IMessage theMsg)
        {
            if(theMsg == null)
                throw new ArgumentNullException("theMsg");

            var msgBytes = theMsg.GetBytes();
            _traceLogger.Info(
                string.Format("Protocol type: {0}, out message:\r\n{1}", MethodBase.GetCurrentMethod().DeclaringType, Encoding.UTF8.GetString(msgBytes))
                );

            // Create a byte array including begin (VT) and end (FS + CR) bytes and serialized message content
            var result = new byte[msgBytes.Length + 3];
            Array.Copy(msgBytes, 0, result, 1, msgBytes.Length);
            result[msgBytes.Length + 1] = Convert.ToByte(28);
            result[msgBytes.Length + 2] = Convert.ToByte(13);
            result[0] = Convert.ToByte(11);
            
            if(_sr != null) 
                _sr.SerializeObject("Messages\\Responce_fs_" + Guid.NewGuid(), new RawDataMessage(result));

            // Return serialized message by this protocol
            return result;
        }

        /// <summary>
        /// Builds message objects from bytes array
        /// </summary>
        /// <param name="theBytes">Bytes array to build message</param>
        /// <returns></returns>
        public virtual IEnumerable<IMessage> BuildMessages(byte[] theBytes)
        {
            var text = Encoding.UTF8.GetString(theBytes);
            _traceLogger.Info(
                string.Format("Protocol type: {0}, BuildMessages:\r\n{1}", MethodBase.GetCurrentMethod().DeclaringType, text)
             );

            //Write all received bytes to the _receiveMemoryStream
            _receiveMemoryStream.Write(theBytes, 0, theBytes.Length);
            //Create a list to collect messages
            var msgList = new List<IMessage>();
            //Read all available messages and add to messages collection
            while (ReadSingleMessage(msgList)){}
            //Return message list
            return msgList;
        }

        private bool ReadSingleMessage(ICollection<IMessage> messages)
        {
            //Go to the beginning of the stream
            _receiveMemoryStream.Position = 0;

            //Return false to wait more bytes from remote application.
            if (_receiveMemoryStream.Length < 1)
            {
                return false;
            }
            //Read fist byte of the message
            var firstByte = BaseTypeSerializer.Byte.Read(_receiveMemoryStream);
            //Check is the first byte is VT byte
            if (firstByte != Convert.ToByte(11))
            {
                Reset();
                throw new InvalidDataException("Can't locate beginning message byte: VT");
            }
            int numberOfControllBytes;
            //search for end message marker FS (28) + CR (13) bytes: 11100+1101
            while (true)
            {
                var num1 = _receiveMemoryStream.ReadByte();
                var num2 = _receiveMemoryStream.ReadByte();
                if (num2 == -1)
                {
                    //If all bytes of the message is not received yet, return to wait more bytes
                    return false;
                }
                if ((ushort) (num1 << 8 | num2) == 7181)
                {
                    var num3 = _receiveMemoryStream.ReadByte();
                    // Check is the next byte is LF (Line Feed = 10) byte
                    if (num3 == 10)
                    {                        
                        //http://www.interfaceware.com/manual/segment_terminator.html
                        _logger.Debug("ReadSingleMessage: vista sent line feed byte at the end of message. Bypass LF byte...");
                        numberOfControllBytes = 4;
                    }
                    else
                    {
                        numberOfControllBytes = 3;
                        if (num3 != -1)
                            _receiveMemoryStream.Position--;
                    }
                    break;
                }
                _receiveMemoryStream.Position--;
            }
            //Read bytes of serialized message and deserialize it
            var msgLength = (int)_receiveMemoryStream.Position - numberOfControllBytes;
            if (msgLength == 0)
            {
                //if no more bytes, return immediately
                if (_receiveMemoryStream.Length == numberOfControllBytes)
                {
                    _receiveMemoryStream = new MemoryStream(); //Clear the stream
                    return false;
                }

                //Create a new memory stream from current except first 3-bytes.
                var bytes = _receiveMemoryStream.ToArray();
                _receiveMemoryStream = new MemoryStream();
                _receiveMemoryStream.Write(bytes, numberOfControllBytes, bytes.Length - numberOfControllBytes);
                return true;
            }
            _receiveMemoryStream.Position = 1; // Exclude fist VT character
            var theMsgBytes = StreamHelper.ReadBytes(_receiveMemoryStream, msgLength);

            var message = _messagesFactory.CreateMessage(theMsgBytes);
            messages.Add(message);
            _traceLogger.Info(
                string.Format("Protocol type: {0}, incoming message:\r\n{1}", MethodBase.GetCurrentMethod().DeclaringType, Encoding.UTF8.GetString(message.GetBytes()))
             );

            //Read remaining bytes to an array
            _receiveMemoryStream.Position += 2 + (numberOfControllBytes - 3); // Exclude FS and CR characters
            var remainingBytes = StreamHelper.ReadBytes(_receiveMemoryStream, (int)(_receiveMemoryStream.Length - (numberOfControllBytes + msgLength)));

            //Re-create the receive memory stream and write remaining bytes
            _receiveMemoryStream = new MemoryStream();
            _receiveMemoryStream.Write(remainingBytes, 0, remainingBytes.Length);

            //Return true to re-call this method to try to read next message
            return (remainingBytes.Length > 1);
        }

        /// <summary>
        /// Resets receiving memory stream.
        /// </summary>
        public void Reset()
        {
            if (_receiveMemoryStream.Length > 0)
            {
                _receiveMemoryStream = new MemoryStream();
            }
        }
    }
}
